<?php
session_start();

// Function to sanitize input data
function sanitizeInput($input) {
    if (is_string($input)) {
        $input = strip_tags($input);
        $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
        return trim($input);
    }
    return $input;
}

// Prevent caching
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

// Redirect if not logged in
if (!isset($_SESSION['status_Account']) || !isset($_SESSION['email']) || $_SESSION['status_Account'] !== 'logged_in') {
    header("Location: ../../index.php");
    exit();
}

include '../../database/db.php';

try {
    // Fetch user data
    $email = $_SESSION['email'];
    $stmt = $connection->prepare("SELECT user_id FROM data WHERE email = ?");
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $user_id = $user['user_id'] ?? null;
    $stmt->close();

    if (!$user_id) {
        header("Location: ../../index.php");
        exit();
    }

    // Fetch appointment data
    $stmt = $connection->prepare("SELECT * FROM appointments WHERE user_id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $appointment = $result->fetch_assoc();
    $stmt->close();

    if (!$appointment) {
        header("Location: ../../fillupform/fillupform.php");
        exit();
    }

    $errors = [];
    $success_message = '';
    $debug_log = [];

    // Define base paths for file uploads and display
    $base_server_path = realpath(__DIR__ . '/../../ProfileImage/image/') . '/';
    $base_relative_url = '../../ProfileImage/image/'; // Used for database storage
    $base_display_url = '../../ProfileImage/image/'; // Absolute URL for image display
    $profile_photo_dir = $base_server_path . 'Profile_Photo/';
    $id_photo_dir = $base_server_path . 'IdPhoto/';

    // Ensure directories exist
    if (!is_dir($profile_photo_dir)) {
        if (!mkdir($profile_photo_dir, 0755, true)) {
            $errors[] = "Failed to create profile photo directory.";
            $debug_log[] = "Failed to create directory: $profile_photo_dir";
        } else {
            $debug_log[] = "Created profile photo directory: $profile_photo_dir";
        }
    }
    if (!is_dir($id_photo_dir)) {
        if (!mkdir($id_photo_dir, 0755, true)) {
            $errors[] = "Failed to create ID photo directory.";
            $debug_log[] = "Failed to create directory: $id_photo_dir";
        } else {
            $debug_log[] = "Created ID photo directory: $id_photo_dir";
        }
    }

    // Function to get next increment for photo filename
    function getNextIncrement($connection, $user_id, $type, $prefix) {
        $stmt = $connection->prepare("SELECT profile_photo, id_photo FROM appointments WHERE user_id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $max_increment = 0;
        while ($row = $result->fetch_assoc()) {
            $path = $type === 'profile' ? $row['profile_photo'] : $row['id_photo'];
            if ($path) {
                $filename = basename($path);
                if (preg_match("/^{$prefix}_{$user_id}_(\d+)\./", $filename, $matches)) {
                    $increment = (int)$matches[1];
                    $max_increment = max($max_increment, $increment);
                }
            }
        }
        $stmt->close();
        return $max_increment + 1;
    }

    // Validate and set current photo URLs from database paths
    $profile_photo_url = '';
    $id_photo_url = '';
    if ($appointment['profile_photo']) {
        $db_profile_path = $appointment['profile_photo'];
        // Handle legacy or incorrect paths
        if (strpos($db_profile_path, '../ProfileImage/image/') !== 0 && strpos($db_profile_path, '/ProfileImage/image/') !== 0) {
            $debug_log[] = "Non-standard profile photo path detected: '$db_profile_path'";
            $filename = basename($db_profile_path);
            $db_profile_path = "../ProfileImage/image/Profile_Photo/$filename";
        }
        // Convert database path to display URL
        $filename = basename($db_profile_path);
        $profile_photo_path = $base_server_path . 'Profile_Photo/' . $filename;
        if (file_exists($profile_photo_path)) {
            $profile_photo_url = $base_display_url . 'Profile_Photo/' . $filename;
            $debug_log[] = "Profile photo found at: '$profile_photo_path', URL: '$profile_photo_url'";
        } else {
            $debug_log[] = "Profile photo not found at: '$profile_photo_path'";
        }
    }
    if ($appointment['id_photo']) {
        $db_id_path = $appointment['id_photo'];
        // Handle legacy or incorrect paths
        if (strpos($db_id_path, '../ProfileImage/image/') !== 0 && strpos($db_id_path, '/ProfileImage/image/') !== 0) {
            $debug_log[] = "Non-standard ID photo path detected: '$db_id_path'";
            $filename = basename($db_id_path);
            $db_id_path = "../ProfileImage/image/IdPhoto/$filename";
        }
        // Convert database path to display URL
        $filename = basename($db_id_path);
        $id_photo_path = $base_server_path . 'IdPhoto/' . $filename;
        if (file_exists($id_photo_path)) {
            $id_photo_url = $base_display_url . 'IdPhoto/' . $filename;
            $debug_log[] = "ID photo found at: '$id_photo_path', URL: '$id_photo_url'";
        } else {
            $debug_log[] = "ID photo not found at: '$id_photo_path'";
        }
    }

    // List of valid regions
    $valid_regions = [
        'Region I – Ilocos Region',
        'Region II – Cagayan Valley',
        'Region III – Central Luzon',
        'Region IV-A – CALABARZON',
        'MIMAROPA Region',
        'Region V – Bicol Region',
        'Region VI – Western Visayas',
        'Region VII – Central Visayas',
        'Region VIII – Eastern Visayas',
        'Region IX – Zamboanga Peninsula',
        'Region X – Northern Mindanao',
        'Region XI – Davao Region',
        'Region XII – SOCCSKSARGEN',
        'Region XIII – Caraga',
        'NCR – National Capital Region',
        'CAR – Cordillera Administrative Region',
        'BARMM – Bangsamoro Autonomous Region in Muslim Mindanao',
        'NIR – Negros Island Region'
    ];

    // Handle form submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Sanitize and validate input
        $first_name = sanitizeInput($_POST['first_name'] ?? '');
        $middle_name = sanitizeInput($_POST['middle_name'] ?? '');
        $last_name = sanitizeInput($_POST['last_name'] ?? '');
        $gender = sanitizeInput($_POST['gender'] ?? '');
        $other_gender = sanitizeInput($_POST['other_gender'] ?? '');
        $birthdate = sanitizeInput($_POST['birthdate'] ?? '');
        $occupation = sanitizeInput($_POST['occupation'] ?? '');
        $address = sanitizeInput($_POST['address'] ?? '');
        $region = sanitizeInput($_POST['region'] ?? '');
        $contact = sanitizeInput($_POST['contact'] ?? '');
        $id_number = sanitizeInput($_POST['id_number'] ?? '');

        // Validate required fields
        if (empty($first_name)) $errors[] = "First name is required.";
        if (empty($last_name)) $errors[] = "Last name is required.";
        if (empty($gender)) $errors[] = "Gender is required.";
        if ($gender === 'Other' && empty($other_gender)) $errors[] = "Please specify gender.";
        if (empty($birthdate)) $errors[] = "Date of birth is required.";
        if (empty($occupation)) $errors[] = "Occupation is required.";
        if (empty($address)) $errors[] = "Address is required.";
        if (empty($region) || !in_array($region, $valid_regions)) $errors[] = "Please select a valid region.";
        if (empty($contact) || !preg_match("/^[0-9]{10}$/", $contact)) $errors[] = "Valid 10-digit contact number is required.";

        // Calculate age
        $birthdate_obj = new DateTime($birthdate);
        $today = new DateTime();
        $age = $today->diff($birthdate_obj)->y;

        // Handle profile photo upload
        $profile_photo = $appointment['profile_photo'];
        if (isset($_FILES['profile_photo']) && $_FILES['profile_photo']['error'] !== UPLOAD_ERR_NO_FILE) {
            if ($_FILES['profile_photo']['error'] === UPLOAD_ERR_OK) {
                $file_type = mime_content_type($_FILES['profile_photo']['tmp_name']);
                $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                if (!in_array($file_type, $allowed_types)) {
                    $errors[] = "Profile photo must be a JPEG, PNG, or GIF image.";
                } elseif ($_FILES['profile_photo']['size'] > 5 * 1024 * 1024) {
                    $errors[] = "Profile photo must be less than 5MB.";
                } else {
                    $ext = pathinfo($_FILES['profile_photo']['name'], PATHINFO_EXTENSION);
                    $increment = getNextIncrement($connection, $user_id, 'profile', 'profile');
                    $profile_filename = "profile_{$user_id}_{$increment}.{$ext}";
                    $profile_destination = $profile_photo_dir . $profile_filename;
                    if (move_uploaded_file($_FILES['profile_photo']['tmp_name'], $profile_destination)) {
                        $profile_photo = $base_relative_url . "Profile_Photo/{$profile_filename}";
                        $_SESSION['profilePhoto'] = $profile_photo;
                        $debug_log[] = "Profile photo uploaded to: '$profile_destination', stored as: '$profile_photo'";
                    } else {
                        $errors[] = "Failed to upload profile photo.";
                        $debug_log[] = "Failed to move profile photo to: '$profile_destination'";
                    }
                }
            } else {
                $errors[] = "Profile photo upload error: " . $_FILES['profile_photo']['error'];
            }
        }

        // Handle ID photo upload
        $id_photo = $appointment['id_photo'];
        if (isset($_FILES['id_photo']) && $_FILES['id_photo']['error'] !== UPLOAD_ERR_NO_FILE) {
            if ($_FILES['id_photo']['error'] === UPLOAD_ERR_OK) {
                $file_type = mime_content_type($_FILES['id_photo']['tmp_name']);
                $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
                if (!in_array($file_type, $allowed_types)) {
                    $errors[] = "ID photo must be a JPEG, PNG, or GIF image.";
                } elseif ($_FILES['id_photo']['size'] > 5 * 1024 * 1024) {
                    $errors[] = "ID photo must be less than 5MB.";
                } else {
                    $ext = pathinfo($_FILES['id_photo']['name'], PATHINFO_EXTENSION);
                    $increment = getNextIncrement($connection, $user_id, 'id', 'id');
                    $id_filename = "id_{$user_id}_{$increment}.{$ext}";
                    $id_destination = $id_photo_dir . $id_filename;
                    if (move_uploaded_file($_FILES['id_photo']['tmp_name'], $id_destination)) {
                        $id_photo = $base_relative_url . "IdPhoto/{$id_filename}";
                        $debug_log[] = "ID photo uploaded to: '$id_destination', stored as: '$id_photo'";
                    } else {
                        $errors[] = "Failed to upload ID photo.";
                        $debug_log[] = "Failed to move ID photo to: '$id_destination'";
                    }
                }
            } else {
                $errors[] = "ID photo upload error: " . $_FILES['id_photo']['error'];
            }
        }

// Update database if no errors
if (empty($errors)) {
    // 1. Update appointments table
    $stmt = $connection->prepare("
        UPDATE appointments SET
            first_name = ?, middle_name = ?, last_name = ?, gender = ?, other_gender = ?,
            birthdate = ?, age = ?, occupation = ?, address = ?, region = ?,
            contact = ?, id_number = ?, profile_photo = ?, id_photo = ?
        WHERE user_id = ?
    ");
    $stmt->bind_param(
        "ssssssisssssssi",
        $first_name, $middle_name, $last_name, $gender, $other_gender,
        $birthdate, $age, $occupation, $address, $region,
        $contact, $id_number, $profile_photo, $id_photo,
        $user_id
    );
    if ($stmt->execute()) {
        $debug_log[] = "Appointments table updated.";
    } else {
        $errors[] = "Failed to update appointments: " . $stmt->error;
    }
    $stmt->close();

    // 2. Update user_information table (for dashboard)
    $stmt2 = $connection->prepare("
        UPDATE user_information SET
            first_name = ?, middle_name = ?, last_name = ?, gender = ?, other_gender = ?,
            birthdate = ?, age = ?, occupation = ?, address = ?, region = ?,
            contact = ?, id_number = ?, profile_photo = ?, id_photo = ?
        WHERE user_id = ?
    ");
    $stmt2->bind_param(
        "ssssssisssssssi",
        $first_name, $middle_name, $last_name, $gender, $other_gender,
        $birthdate, $age, $occupation, $address, $region,
        $contact, $id_number, $profile_photo, $id_photo,
        $user_id
    );
    if ($stmt2->execute()) {
        $debug_log[] = "User information table updated.";
    } else {
        $errors[] = "Failed to update user_information: " . $stmt2->error;
    }
    $stmt2->close();

    $success_message = "Profile updated successfully.";
}

        // Update photo URLs after successful upload
        if (empty($errors) && $profile_photo !== $appointment['profile_photo']) {
            $profile_photo_path = $base_server_path . 'Profile_Photo/' . $profile_filename;
            if (file_exists($profile_photo_path)) {
                $profile_photo_url = $base_display_url . 'Profile_Photo/' . $profile_filename;
                $debug_log[] = "Profile photo URL updated to: '$profile_photo_url'";
            } else {
                $debug_log[] = "Profile photo not found after upload at: '$profile_photo_path'";
            }
        }
        if (empty($errors) && $id_photo !== $appointment['id_photo']) {
            $id_photo_path = $base_server_path . 'IdPhoto/' . $id_filename;
            if (file_exists($id_photo_path)) {
                $id_photo_url = $base_display_url . 'IdPhoto/' . $id_filename;
                $debug_log[] = "ID photo URL updated to: '$id_photo_url'";
            } else {
                $debug_log[] = "ID photo not found after upload at: '$id_photo_path'";
            }
        }
    }

    // Re-fetch appointment data to display updated values
    $stmt = $connection->prepare("SELECT * FROM appointments WHERE user_id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $appointment = $result->fetch_assoc();
    $stmt->close();

} catch (Exception $e) {
    $errors[] = "Database error: " . $e->getMessage();
    $debug_log[] = "Exception: " . $e->getMessage();
} finally {
    $connection->close();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
    <meta http-equiv="Pragma" content="no-cache">
    <meta http-equiv="Expires" content="0">
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link rel="icon" type="image/x-icon" href="/image/icons/logo1.ico">
    <link rel="stylesheet" href="../../css/style_user1.css">
    <link rel="stylesheet" href="../../css/edit_profile1.css">
    <title>Edit Profile</title>
</head>
<body>
    <div class="sidebar">
        <button class="close-sidebar" aria-label="Close Sidebar">
            <i class='bx bx-x'></i>
        </button>
        <img src="../../image/icons/logo1.ico" alt="Organization Logo" class="logo">
        <h2>Dashboard</h2>

        <a href="../dashboard.php" class="<?= basename($_SERVER['PHP_SELF']) == 'dashboard.php' ? 'active' : '' ?>">Dashboard</a>
        <a href="edit_profile.php" class="<?= basename($_SERVER['PHP_SELF']) == 'edit_profile.php' ? 'active' : '' ?>">Edit Profile</a>

        <a href="../appointments/appointment_table.php" class="<?= basename($_SERVER['PHP_SELF']) == 'appointment_table.php' ? 'active' : '' ?>">Appointments</a>

        <a href="../../logout.php">Logout</a>
    </div>
    <div class="main-content">
        <div class="dashboard-header">
            <button class="menu-toggle" aria-label="Toggle Sidebar" aria-expanded="false">
                <i class='bx bx-menu'></i>
            </button>
            <h1>Edit Profile</h1>
        </div>
        <div class="card">
            <h2>Profile Information</h2>
            <?php if ($success_message): ?>
                <div class="success"><?php echo htmlspecialchars($success_message); ?></div>
            <?php endif; ?>
            <?php if (!empty($errors)): ?>
                <div class="error">
                    <?php foreach ($errors as $error): ?>
                        <p><?php echo htmlspecialchars($error); ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            <form method="POST" enctype="multipart/form-data">
                <div class="form-grid">
                    <div class="form-group">
                        <label for="first_name">First Name *</label>
                        <input type="text" id="first_name" name="first_name" value="<?php echo htmlspecialchars($appointment['first_name'] ?? ''); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="middle_name">Middle Name</label>
                        <input type="text" id="middle_name" name="middle_name" value="<?php echo htmlspecialchars($appointment['middle_name'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label for="last_name">Last Name *</label>
                        <input type="text" id="last_name" name="last_name" value="<?php echo htmlspecialchars($appointment['last_name'] ?? ''); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="gender">Gender *</label>
                        <select id="gender" name="gender" required>
                            <option value="Male" <?php echo ($appointment['gender'] ?? '') === 'Male' ? 'selected' : ''; ?>>Male</option>
                            <option value="Female" <?php echo ($appointment['gender'] ?? '') === 'Female' ? 'selected' : ''; ?>>Female</option>
                            <option value="Other" <?php echo ($appointment['gender'] ?? '') === 'Other' ? 'selected' : ''; ?>>Other</option>
                        </select>
                    </div>
                    <div class="form-group" id="other_gender_group" style="display: <?php echo ($appointment['gender'] ?? '') === 'Other' ? 'block' : 'none'; ?>;">
                        <label for="other_gender">Specify Gender *</label>
                        <input type="text" id="other_gender" name="other_gender" value="<?php echo htmlspecialchars($appointment['other_gender'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label for="birthdate">Date of Birth *</label>
                        <input type="date" id="birthdate" name="birthdate" value="<?php echo htmlspecialchars($appointment['birthdate'] ?? ''); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="occupation">Occupation *</label>
                        <input type="text" id="occupation" name="occupation" value="<?php echo htmlspecialchars($appointment['occupation'] ?? ''); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="address">Address *</label>
                        <textarea id="address" name="address" required><?php echo htmlspecialchars($appointment['address'] ?? ''); ?></textarea>
                    </div>
                    <div class="form-group">
                        <label for="region">Region *</label>
                        <select id="region" name="region" required aria-describedby="region-help">
                            <option value="" disabled <?php echo empty($appointment['region']) ? 'selected' : ''; ?>>Select a Region</option>
                            <?php foreach ($valid_regions as $region_option): ?>
                                <option value="<?php echo htmlspecialchars($region_option); ?>" <?php echo ($appointment['region'] ?? '') === $region_option ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($region_option); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <small id="region-help" class="form-text">Please select your region from the list.</small>
                    </div>
                    <div class="form-group">
                        <label for="contact">Contact Number * (+63)</label>
                        <input type="number" id="contact" name="contact" value="<?php echo htmlspecialchars($appointment['contact'] ?? ''); ?>" pattern="[0-9]{10}" maxlength="10" required>
                    </div>
                    <div class="form-group">
                        <label for="id_number">ID Number</label>
                        <input type="text" id="id_number" name="id_number" value="<?php echo htmlspecialchars($appointment['id_number'] ?? ''); ?>">
                    </div>
                    <div class="form-group">
                        <label for="profile_photo">Profile Photo (JPEG, PNG, GIF, max 5MB)</label>
                        <div class="photo-preview-container">
                            <?php if ($profile_photo_url): ?>
                                <img src="<?php echo htmlspecialchars($profile_photo_url); ?>" alt="Profile Photo Preview" class="photo-preview" id="profile_preview">
                            <?php else: ?>
                                <div class="photo-placeholder" id="profile_preview">No Photo Available</div>
                            <?php endif; ?>
                        </div>
                        <input type="file" id="profile_photo" name="profile_photo" accept="image/jpeg,image/png,image/gif">
                    </div>
                    <div class="form-group">
                        <label for="id_photo">ID Photo (JPEG, PNG, GIF, max 5MB)</label>
                        <div class="photo-preview-container">
                            <?php if ($id_photo_url): ?>
                                <img src="<?php echo htmlspecialchars($id_photo_url); ?>" alt="ID Photo Preview" class="photo-preview" id="id_preview">
                            <?php else: ?>
                                <div class="photo-placeholder" id="id_preview">No Photo Available</div>
                            <?php endif; ?>
                        </div>
                        <input type="file" id="id_photo" name="id_photo" accept="image/jpeg,image/png,image/gif">
                    </div>
                </div>
                <button type="submit" class="submit-btn">Save Changes</button>
            </form>
        </div>
    </div>

    <script src="../../scripts/edit_profile1.js"></script>
</body>
</html>