<?php
include '../database/db.php';
session_start();

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', 'php_error.log');

// Prevent caching
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

// CSRF token generation
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Redirect if not logged in
if (empty($_SESSION['status_Account']) || empty($_SESSION['email'])) {
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'error' => 'User not logged in.']);
        exit();
    }
    header("Location: ../index.php");
    exit();
}

// Fetch user_id
$email = $_SESSION['email'];
$stmt = $connection->prepare("SELECT user_id FROM data WHERE email = ?");
$stmt->bind_param("s", $email);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$user_id = $user['user_id'] ?? null;
$stmt->close();

if (!$user_id) {
    header("Location: ../index.php");
    exit();
}

$errors = [];
$success_message = '';
$debug_log = [];

function logError($message) {
    file_put_contents('error_log.txt', date('Y-m-d H:i:s') . ": $message\n", FILE_APPEND);
}

// Directory paths
define('PHYSICAL_UPLOAD_DIR', '../ProfileImage/image/IdPhoto/');
define('RELATIVE_UPLOAD_DIR', '../ProfileImage/image/IdPhoto/');
define('PROFILE_PHYSICAL_UPLOAD_DIR', '../ProfileImage/image/Profile_Photo/');
define('PROFILE_RELATIVE_UPLOAD_DIR', '../ProfileImage/image/Profile_Photo/');

// List of valid regions
$valid_regions = [
    'Region I – Ilocos Region',
    'Region II – Cagayan Valley',
    'Region III – Central Luzon',
    'Region IV-A – CALABARZON',
    'MIMAROPA Region',
    'Region V – Bicol Region',
    'Region VI – Western Visayas',
    'Region VII – Central Visayas',
    'Region VIII – Eastern Visayas',
    'Region IX – Zamboanga Peninsula',
    'Region X – Northern Mindanao',
    'Region XI – Davao Region',
    'Region XII – SOCCSKSARGEN',
    'Region XIII – Caraga',
    'NCR – National Capital Region',
    'CAR – Cordillera Administrative Region',
    'BARMM – Bangsamoro Autonomous Region in Muslim Mindanao',
    'NIR – Negros Island Region'
];

// Function to generate a unique appointment ID
function generateAppointmentId($connection) {
    $maxRetries = 5;
    $attempt = 0;

    while ($attempt < $maxRetries) {
        $randomNumber = mt_rand(0, 99999999);
        $appointment_id = sprintf("%08d", $randomNumber);
        $stmt = $connection->prepare("SELECT COUNT(*) AS count FROM appointments WHERE appointment_id = ?");
        $stmt->bind_param("s", $appointment_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();

        if ($row['count'] == 0) {
            return $appointment_id;
        }

        $attempt++;
    }

    throw new Exception("Unable to generate a unique appointment ID after $maxRetries attempts.");
}

// For AJAX requests, ensure proper JSON responses
if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
    header('Content-Type: application/json');
    
    // Set error handler for AJAX requests
    set_error_handler(function($errno, $errstr, $errfile, $errline) {
        echo json_encode([
            'success' => false,
            'error' => 'Server error occurred',
            'debug' => "$errstr in $errfile on line $errline"
        ]);
        exit;
    });
}

// Handle ID photo upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['idPhoto']) && isset($_POST['upload_id_photo'])) {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $errors['general'] = 'Invalid CSRF token.';
        $debug_log[] = 'CSRF token validation failed.';
        logError('CSRF token validation failed for ID photo upload.');
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
            exit();
        }
    }

    $id_type = filter_input(INPUT_POST, 'idType', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $id_number = filter_input(INPUT_POST, 'idNumber', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $file = $_FILES['idPhoto'];
    $validTypes = ['image/jpeg', 'image/jpg'];

    $baseFileName = 'id_' . $user_id;
    $fileName = $baseFileName . '.jpg';
    $uploadPath = PHYSICAL_UPLOAD_DIR . $fileName;
    $relativePath = RELATIVE_UPLOAD_DIR . $fileName;

    $counter = 1;
    while (file_exists($uploadPath)) {
        $fileName = $baseFileName . '_' . $counter . '.jpg';
        $uploadPath = PHYSICAL_UPLOAD_DIR . $fileName;
        $relativePath = RELATIVE_UPLOAD_DIR . $fileName;
        $counter++;
    }

    if (!is_dir(PHYSICAL_UPLOAD_DIR)) {
        if (!mkdir(PHYSICAL_UPLOAD_DIR, 0777, true)) {
            $errors['idPhoto'] = 'Failed to create upload directory.';
            $debug_log[] = "Failed to create directory: " . PHYSICAL_UPLOAD_DIR;
            logError("Failed to create upload directory: " . PHYSICAL_UPLOAD_DIR);
        }
    }

    if (empty($id_type)) {
        $errors['idType'] = "Please select a valid ID type.";
    } elseif (empty($id_number) || !preg_match('/^[A-Za-z0-9-]{1,50}$/', $id_number)) {
        $errors['idNumber'] = "ID number must be alphanumeric (up to 50 characters).";
    } elseif (!in_array($file['type'], $validTypes)) {
        $errors['idPhoto'] = "Please upload a valid .jpg or .jpeg file.";
    } elseif ($file['error'] !== UPLOAD_ERR_OK) {
        $errors['idPhoto'] = "Upload failed: " . [
            UPLOAD_ERR_INI_SIZE => "File exceeds server size limit.",
            UPLOAD_ERR_FORM_SIZE => "File exceeds form size limit.",
            UPLOAD_ERR_PARTIAL => "File was only partially uploaded.",
            UPLOAD_ERR_NO_FILE => "No file was uploaded.",
            UPLOAD_ERR_NO_TMP_DIR => "Temporary folder missing.",
            UPLOAD_ERR_CANT_WRITE => "Failed to write file to disk.",
            UPLOAD_ERR_EXTENSION => "A PHP extension stopped the upload."
        ][$file['error']] ?? "Unknown error.";
    } elseif (!is_uploaded_file($file['tmp_name'])) {
        $errors['idPhoto'] = "Security error: Invalid file upload.";
    } else {
        $imageInfo = @getimagesize($file['tmp_name']);
        if ($imageInfo === false) {
            $errors['idPhoto'] = "Invalid image file. Please upload a valid JPEG.";
        } elseif ($imageInfo[0] < 180 || $imageInfo[1] < 180 || abs($imageInfo[0] - $imageInfo[1]) > 20) {
            $errors['idPhoto'] = "Image must be approximately 2x2 inches (~192x192 pixels at 96 DPI).";
        } else {
            if (extension_loaded('gd') && function_exists('imagecreatefromjpeg')) {
                $image = @imagecreatefromjpeg($file['tmp_name']);
                if ($image === false) {
                    $errors['idPhoto'] = "Failed to process image. Try another file.";
                } else {
                    if (imagejpeg($image, $uploadPath, 75)) {
                        imagedestroy($image);
                        $_SESSION['idPhoto'] = $relativePath;
                        $_SESSION['idType'] = $id_type;
                        $_SESSION['idNumber'] = $id_number;
                        $debug_log[] = "ID photo uploaded to: $uploadPath, stored as: $relativePath";
                        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                            header('Content-Type: application/json');
                            echo json_encode(['success' => true, 'message' => 'ID uploaded successfully.', 'idPhoto' => $relativePath]);
                            session_write_close();
                            exit();
                        }
                    } else {
                        imagedestroy($image);
                        $errors['idPhoto'] = "Failed to save image to server.";
                        $debug_log[] = "Failed to save image to: $uploadPath";
                        logError("Failed to save ID photo to: $uploadPath");
                    }
                }
            } else {
                if (!empty($_POST['compressedImage'])) {
                    $data = $_POST['compressedImage'];
                    $data = str_replace('data:image/jpeg;base64,', '', $data);
                    $data = base64_decode($data);
                    if ($data === false) {
                        $errors['idPhoto'] = "Invalid compressed image data.";
                    } elseif (file_put_contents($uploadPath, $data)) {
                        $_SESSION['idPhoto'] = $relativePath;
                        $_SESSION['idType'] = $id_type;
                        $_SESSION['idNumber'] = $id_number;
                        $debug_log[] = "Compressed ID photo uploaded to: $uploadPath, stored as: $relativePath";
                        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                            header('Content-Type: application/json');
                            echo json_encode(['success' => true, 'message' => 'ID uploaded successfully.', 'idPhoto' => $relativePath]);
                            session_write_close();
                            exit();
                        }
                    } else {
                        $errors['idPhoto'] = "Failed to save compressed image to server.";
                        $debug_log[] = "Failed to save compressed image to: $uploadPath";
                        logError("Failed to save compressed ID photo to: $uploadPath");
                    }
                } else {
                    $errors['idPhoto'] = "Server image processing unavailable.";
                }
            }
        }
    }

    if (!empty($errors) && !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
        exit();
    }
}

// Handle photo removal
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['removeIdPhoto'])) {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $errors['general'] = 'Invalid CSRF token.';
        $debug_log[] = 'CSRF token validation failed for photo removal.';
        logError('CSRF token validation failed for photo removal.');
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
            exit();
        }
    }

    if (!empty($_SESSION['idPhoto'])) {
        $fullPath = str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto']);
        if (file_exists($fullPath)) {
            unlink($fullPath);
            $debug_log[] = "Removed ID photo: $fullPath";
        } else {
            $debug_log[] = "ID photo not found for removal: $fullPath";
        }
    }
    unset($_SESSION['idPhoto'], $_SESSION['idType'], $_SESSION['idNumber']);
    $debug_log[] = "Cleared ID photo session data.";
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'message' => 'ID photo removed successfully.']);
        session_write_close();
        exit();
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_appointment'])) {
    try {
        if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
            $errors['general'] = 'Invalid CSRF token.';
            $debug_log[] = 'CSRF token validation failed for form submission.';
            logError('CSRF token validation failed for form submission.');
            if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
                exit();
            }
        }

        function sanitizeInput($input) {
            if (is_string($input)) {
                $input = strip_tags($input);
                $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
                return trim($input);
            }
            return $input;
        }

        $last_name = sanitizeInput($_POST['lastName'] ?? '');
        $first_name = sanitizeInput($_POST['firstName'] ?? '');
        $middle_name = !empty($_POST['middleName']) ? sanitizeInput($_POST['middleName']) : null;
        $gender = sanitizeInput($_POST['gender'] ?? '');
        $other_gender = !empty($_POST['othergender']) ? sanitizeInput($_POST['othergender']) : null;
        $birthdate = sanitizeInput($_POST['birthdate'] ?? '');
        $occupation = sanitizeInput($_POST['occupation'] ?? '');
      	$street = sanitizeInput($_POST['street'] ?? '');
      	$barangay = sanitizeInput($_POST['barangay'] ?? '');
      	$city = sanitizeInput($_POST['city'] ?? '');
      	$province = sanitizeInput($_POST['province'] ?? '');
        $address = sanitizeInput($_POST['address'] ?? '');
        $region = sanitizeInput($_POST['region'] ?? '');
        $contact = sanitizeInput($_POST['contact'] ?? '');
        $appointment_date = sanitizeInput($_POST['appointmentDate'] ?? '');
        $appointment_time = sanitizeInput($_POST['appointmentTime'] ?? '');
        $purpose = sanitizeInput($_POST['purpose'] ?? '');

        $email = $_SESSION['email']; // Use session email directly
        $profile_photo = $_SESSION['profilePhoto'] ?? '';
        $id_type = $_SESSION['idType'] ?? '';
        $id_number = $_SESSION['idNumber'] ?? '';
        $id_photo = $_SESSION['idPhoto'] ?? '';

        // Validate inputs
        if (empty($last_name)) $errors['lastName'] = 'Last name is required.';
        if (empty($first_name)) $errors['firstName'] = 'First name is required.';
        if (empty($gender)) $errors['gender'] = 'Gender is required.';
        if ($gender === 'Other' && empty($other_gender)) $errors['othergender'] = 'Please specify your gender.';
        if (empty($birthdate) || !DateTime::createFromFormat('Y-m-d', $birthdate)) $errors['birthdate'] = 'Valid date of birth is required.';
        if (empty($occupation)) $errors['occupation'] = 'Occupation is required.';
      	if (empty($street)) $errors['address'] = 'Street is required.';
      	if (empty($barangay)) $errors['address'] = 'Barangay is required.';
      	if (empty($city)) $errors['address'] = 'City/Municipality is required.';
      	if (empty($province)) $errors['address'] = 'Province is required.';
        if (empty($address)) $errors['address'] = 'Address is required.';
        if (empty($region) || !in_array($region, $valid_regions)) $errors['region'] = 'Please select a valid region.';
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Valid email is required.';
        if (empty($contact) || !preg_match('/^[0-9]{10}$/', $contact)) $errors['contact'] = 'Valid 10-digit contact number is required.';
        if (empty($appointment_date)) $errors['appointmentDate'] = 'Appointment date is required.';
        if (empty($appointment_time)) {
            $errors['appointmentTime'] = 'Appointment time is required.';
        } elseif (!preg_match('/^([0-1][0-9]|2[0-2]):[0-5][0-9]$/', $appointment_time) || $appointment_time < '05:00' || $appointment_time > '22:00') {
            $errors['appointmentTime'] = 'Appointment time must be between 5:00 AM and 10:00 PM.';
            $debug_log[] = "Invalid appointment time: $appointment_time";
            logError("Invalid appointment time: $appointment_time");
        }
        if (empty($purpose)) $errors['purpose'] = 'Purpose is required.';
        if (empty($profile_photo)) {
            $errors['myFile'] = 'Profile photo is required.';
            $debug_log[] = "Profile photo missing: $profile_photo";
        } elseif (!file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $profile_photo))) {
            $errors['myFile'] = 'Profile photo file not found.';
            $debug_log[] = "Profile photo not found: " . str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $profile_photo);
            logError("Profile photo not found: " . str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $profile_photo));
        }
        if (empty($id_type)) $errors['idType'] = 'ID type is required.';
        if (empty($id_number)) $errors['idNumber'] = 'ID number is required.';
        if (empty($id_photo)) {
            $errors['idPhoto'] = 'ID photo is required.';
            $debug_log[] = "ID photo missing: $id_photo";
        } elseif (!file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $id_photo))) {
            $errors['idPhoto'] = 'ID photo file not found.';
            $debug_log[] = "ID photo not found: " . str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $id_photo);
            logError("ID photo not found: " . str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $id_photo));
        }

        if (!empty($appointment_date)) {
            $date = DateTime::createFromFormat('Y-m-d', $appointment_date);
            if ($date === false || $date->format('N') !== '4') {
                $errors['appointmentDate'] = 'Appointment date must be a Thursday.';
            }
        }

        if (!empty($errors)) {
            if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                echo json_encode([
                    'success' => false,
                    'errors' => $errors,
                    'debug' => $debug_log
                ]);
                exit;
            }
        } else {
            // Database operations...
            $connection->begin_transaction();
            try {
                // Generate unique appointment ID
                $appointment_id = generateAppointmentId($connection);

                // Calculate age from birthdate
                $birthdate_obj = new DateTime($birthdate);
                $today = new DateTime();
                $age = $birthdate_obj->diff($today)->y;

                // Prepare user information statement
                $userStmt = $connection->prepare("
                    INSERT INTO user_information (
                        user_id, last_name, first_name, middle_name, gender, other_gender,
                        birthdate, age, occupation, address, region, email, contact,
                        profile_photo, id_type, id_number, id_photo
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                        last_name = VALUES(last_name),
                        first_name = VALUES(first_name),
                        middle_name = VALUES(middle_name),
                        gender = VALUES(gender),
                        other_gender = VALUES(other_gender),
                        birthdate = VALUES(birthdate),
                        age = VALUES(age),
                        occupation = VALUES(occupation),
                        address = VALUES(address),
                        region = VALUES(region),
                        contact = VALUES(contact),
                        profile_photo = VALUES(profile_photo),
                        id_type = VALUES(id_type),
                        id_number = VALUES(id_number),
                        id_photo = VALUES(id_photo)
                ");

                if (!$userStmt) {
                    throw new Exception("Prepare failed: " . $connection->error);
                }

                $userStmt->bind_param(
                    "sssssssssssssssss",
                    $user_id, $last_name, $first_name, $middle_name, $gender, $other_gender,
                    $birthdate, $age, $occupation, $address, $region, $email, $contact,
                    $profile_photo, $id_type, $id_number, $id_photo
                );

                if (!$userStmt->execute()) {
                    throw new Exception("Error updating user information: " . $userStmt->error);
                }

                // Prepare appointments statement
                $apptStmt = $connection->prepare("
                    INSERT INTO appointments (
                        appointment_id, user_id, last_name, first_name, middle_name,
                        gender, other_gender, birthdate, age, occupation, address,
                        region, email, contact, appointment_date, appointment_time,
                        purpose, profile_photo, id_type, id_number, id_photo, status
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pending')
                ");

                if (!$apptStmt) {
                    throw new Exception("Prepare failed: " . $connection->error);
                }

                $apptStmt->bind_param(
                    "sissssssissssssssssss",
                    $appointment_id, $user_id, $last_name, $first_name, $middle_name,
                    $gender, $other_gender, $birthdate, $age, $occupation, $address,
                    $region, $email, $contact, $appointment_date, $appointment_time,
                    $purpose, $profile_photo, $id_type, $id_number, $id_photo
                );

                if (!$apptStmt->execute()) {
                    throw new Exception("Error inserting appointment: " . $apptStmt->error);
                }

                $connection->commit();
                
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => true,
                    'message' => "Appointment submitted successfully with ID: $appointment_id"
                ]);
                exit();

            } catch (Exception $e) {
                $connection->rollback();
                error_log("Database error: " . $e->getMessage());
                
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'error' => 'Database error occurred. Please try again.',
                    'debug' => $e->getMessage()
                ]);
                exit();
            }
        }
    } catch (Exception $e) {
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => 'Server error: ' . $e->getMessage(),
                'debug' => $debug_log
            ]);
            exit;
        }
        throw $e;
    }
}

$connection->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
    <meta http-equiv="Pragma" content="no-cache">
    <meta http-equiv="Expires" content="0">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css" media="all">
    <link rel="icon" type="image/x-icon" href="../image/icons/logo1.ico">
    <link rel="stylesheet" href="/assets/css/fillupform1.css" />
    <title>Registration Form</title>
</head>
<body>
    <header class="form-header">
            <img src="../image/icons/logo1.ico" alt="Organization Logo" class="logo">
            <h1 id="form-title">Registration Form</h1>
            <a href="../logout.php" class="logout-link">Logout</a>
    </header>
    <div class="form-container">
        <form id="registrationForm" action="" method="POST" enctype="multipart/form-data" aria-label="Registration Form">
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">

            <!-- Photo Upload Section -->
            <div class="form-section">
                <h2>Photo Identification</h2>
                <div class="form-group">
                    <label for="myFile" class="label required">Profile Photo (2x2, JPG/JPEG)</label>
                    <div class="photo-upload-group">
                        <div class="photo-placeholder" id="profilePhotoPreview" aria-live="polite">
                            <img id="profilePhotoImg" class="photo-preview <?php echo !empty($_SESSION['profilePhoto']) && file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $_SESSION['profilePhoto'])) ? 'active' : ''; ?>" src="<?php echo htmlspecialchars($_SESSION['profilePhoto'] ?? ''); ?>" alt="Profile Photo Preview" aria-hidden="<?php echo empty($_SESSION['profilePhoto']) || !file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $_SESSION['profilePhoto'])) ? 'true' : 'false'; ?>">
                            <?php if (empty($_SESSION['profilePhoto']) || !file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $_SESSION['profilePhoto']))): ?>
                                <span>No Photo Uploaded</span>
                            <?php endif; ?>
                        </div>
                        <a href="../photo_config_upload/photo_upload.php" class="btn btn-primary" aria-label="Upload profile photo">Upload Profile Photo</a>
                        <div class="photo-upload-note">JPG/JPEG, Max 2MB, 192x192 pixels</div>
                    </div>
                    <span class="error" id="myFile-error"><?php echo htmlspecialchars($errors['myFile'] ?? ''); ?></span>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="idType" class="label required">Valid ID Type</label>
                        <select id="idType" name="idType" required aria-required="true">
                            <option value="" <?php echo empty($_SESSION['idType']) ? 'selected' : ''; ?>>Select ID Type</option>
                            <?php
                            $idTypes = [
                                "Professional Regulation Commission", "Government Service Insurance System", "Passport", "SSS ID", "Drivers License",
                                "Overseas Workers Welfare Administration", "Senior Citizen ID", "NBI Clearance", "Unified Multi-purpose Identification (UMID) Card",
                                "Voters ID", "TIN ID", "PhilHealth ID", "Postal ID", "Seamans Book", "Philippine Identification Card",
                                "Philippine Passport", "Philippine Postal ID", "Police Clearance", "Barangay Clearance", "Integrated Bar of the Philippines",
                                "National ID", "Philippine Identification (PhilID)/ePhilID", "School ID", "Alien Certification"
                            ];
                            foreach ($idTypes as $type) {
                                $selected = (isset($_SESSION['idType']) && $_SESSION['idType'] === $type) ? 'selected' : '';
                                echo "<option value=\"" . htmlspecialchars($type) . "\" $selected>" . htmlspecialchars($type) . "</option>";
                            }
                            ?>
                        </select>
                        <span class="error" id="idType-error"><?php echo htmlspecialchars($errors['idType'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="idNumber" class="label required">ID Number</label>
                        <input type="text" id="idNumber" name="idNumber" required autocomplete="off" aria-required="true" value="<?php echo htmlspecialchars($_SESSION['idNumber'] ?? ''); ?>" placeholder="Example: xxxx-xxxx-xxxx-xxxx" maxlength="50">
                        <span class="error" id="idNumber-error"><?php echo htmlspecialchars($errors['idNumber'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="idPhoto" class="label required">ID Photo (2x2, JPG/JPEG)</label>
                    <div class="photo-upload-group">
                        <div class="photo-placeholder" id="idPhotoPreview" aria-live="polite">
                            <img id="idPhotoImg" class="photo-preview <?php echo !empty($_SESSION['idPhoto']) && file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto'])) ? 'active' : ''; ?>" src="<?php echo !empty($_SESSION['idPhoto']) ? htmlspecialchars($_SESSION['idPhoto']) : ''; ?>" alt="ID Photo Preview" aria-hidden="<?php echo empty($_SESSION['idPhoto']) || !file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto'])) ? 'true' : 'false'; ?>">
                            <?php if (empty($_SESSION['idPhoto']) || !file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto']))): ?>
                                <span>No ID Photo Uploaded</span>
                            <?php endif; ?>
                        </div>
                        <div class="action-buttons" id="actionButtons">
                            <button type="button" class="btn btn-primary" id="chooseIdPhotoBtn" aria-label="Choose ID photo file">Select ID Photo</button>
                            <?php if (!empty($_SESSION['idPhoto']) && file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto']))): ?>
                                <button type="button" class="btn btn-danger" id="removeIdPhotoBtn" aria-label="Remove uploaded ID photo">Remove ID Photo</button>
                            <?php endif; ?>
                        </div>
                        <input type="file" id="idPhotoInput" name="idPhoto" accept="image/jpeg,image/jpg" style="display: none;" aria-label="Select ID photo">
                        <input type="hidden" id="compressedImage" name="compressedImage">
                        <div class="photo-upload-note">JPG/JPEG, Max 2MB, 192x192 pixels. Photo uploads automatically after selection.</div>
                        <div class="progress-bar" id="idPhotoProgress"><div></div></div>
                        <div class="loading-spinner" id="idPhotoSpinner" aria-label="Processing"></div>
                        <div class="success-message" id="idPhotoSuccess" role="alert"><?php echo !empty($_SESSION['idPhoto']) && file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto'])) ? 'ID uploaded successfully.' : ''; ?></div>
                    </div>
                    <span class="error" id="idPhoto-error"><?php echo htmlspecialchars($errors['idPhoto'] ?? ''); ?></span>
                </div>
            </div>

            <!-- Personal Information Section -->
            <div class="form-section">
                <h2>Personal Information</h2>
                <div class="form-row">
                    <div class="form-group">
                        <label for="lastName" class="label required">Last Name</label>
                        <input type="text" id="lastName" name="lastName" required autocomplete="off" aria-required="true" placeholder="Enter your last name">
                        <span class="error" id="lastName-error"><?php echo htmlspecialchars($errors['lastName'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="firstName" class="label required">First Name</label>
                        <input type="text" id="firstName" name="firstName" required autocomplete="off" aria-required="true" placeholder="Enter your first name">
                        <span class="error" id="firstName-error"><?php echo htmlspecialchars($errors['firstName'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="middleName" class="label">Middle Name (Optional)</label>
                    <input type="text" id="middleName" name="middleName" autocomplete="off" placeholder="Enter your middle name">
                    <span class="error" id="middleName-error"><?php echo htmlspecialchars($errors['middleName'] ?? ''); ?></span>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gender" class="label required">Gender</label>
                        <select id="gender" name="gender" required aria-required="true">
                            <option value="" selected disabled>Select Gender</option>
                            <option value="Male">Male</option>
                            <option value="Female">Female</option>
                            <option value="Other">Other</option>
                        </select>
                        <span class="error" id="gender-error"><?php echo htmlspecialchars($errors['gender'] ?? ''); ?></span>
                    </div>
                    <div class="form-group" id="othergenderGroup" style="display: none;">
                        <label for="othergender" class="label required">Specify Gender</label>
                        <input type="text" id="othergender" name="othergender" autocomplete="off" placeholder="Specify your gender">
                        <span class="error" id="othergender-error"><?php echo htmlspecialchars($errors['othergender'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="birthdate" class="label required">Date of Birth</label>
                        <input type="date" id="birthdate" name="birthdate" required autocomplete="off" max="<?php echo date('Y-m-d'); ?>" aria-required="true">
                        <span class="error" id="birthdate-error"><?php echo htmlspecialchars($errors['birthdate'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="age" class="label required">Age</label>
                        <input type="number" id="age" name="age" required autocomplete="off" min="1" max="120" aria-required="true" placeholder="Auto-calculated">
                        <span class="error" id="age-error"><?php echo htmlspecialchars($errors['age'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="occupation" class="label required">Occupation</label>
                    <input type="text" id="occupation" name="occupation" required autocomplete="off" aria-required="true" placeholder="Enter your occupation">
                    <span class="error" id="occupation-error"><?php echo htmlspecialchars($errors['occupation'] ?? ''); ?></span>
                </div>
            </div>

            <!-- Contact and Address Section -->
            <div class="form-section">
                <h2>Contact and Address</h2>
                <div class="form-group">
                    <label for="street" class="label required">Street</label>
                    <input type="text" id="street" name="street">
                    <label for="barangay" class="label required">Barangay</label>
                    <input type="text" id="barangay" name="barangay">
                    <label for="city" class="label required">City/Municipality</label>
                    <input type="text" id="city" name="city">
                    <label for="province" class="label required">Province</label>
                    <input type="text" id="province" name="province">
                    <label for="address" class="label required">Full Address</label>
                    <input type="text" id="address" name="address" required autocomplete="off" aria-required="true" readonly>
                    <span class="error" id="address-error"><?php echo htmlspecialchars($errors['address'] ?? ''); ?></span>
                </div>
              <div class="form-group">
                    <label for="region" class="label required">Region</label>
                    <select id="region" name="region" required aria-required="true" aria-describedby="region-help">
                        <option value="" selected disabled>Select a Region</option>
                        <?php foreach ($valid_regions as $region_option): ?>
                            <option value="<?php echo htmlspecialchars($region_option); ?>">
                                <?php echo htmlspecialchars($region_option); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <div class="photo-upload-note">Please select your region from the list.</div>
                    <span class="error" id="region-error"><?php echo htmlspecialchars($errors['region'] ?? ''); ?></span>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="email" class="label required">Email Address</label>
                        <input type="email" id="email" name="email" required readonly autocomplete="off" aria-required="true" value="<?php echo htmlspecialchars($_SESSION['email']); ?>">
                        <span class="error" id="email-error"><?php echo htmlspecialchars($errors['email'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="contact" class="label required">Contact Number</label>
                        <div class="contact-group">
                            <input type="text" value="+63" readonly aria-label="Country code">
                            <input type="text" id="contact" name="contact" required autocomplete="off" pattern="[0-9]{10}" title="Please enter a valid 10-digit phone number" placeholder="9123456789" aria-required="true">
                        </div>
                        <span class="error" id="contact-error"><?php echo htmlspecialchars($errors['contact'] ?? ''); ?></span>
                    </div>
                </div>
            </div>

            <!-- Appointment Details Section -->
            <div class="form-section">
                <h2>Appointment Details</h2>
                <div class="form-row">
                    <div class="form-group">
                        <label for="appointmentDate" class="label required">Appointment Date (Thursdays)</label>
                        <input type="text" id="appointmentDate" name="appointmentDate" required autocomplete="off" aria-required="true" placeholder="Select a Thursday">
                        <div class="photo-upload-note">Only Thursdays are selectable</div>
                        <span class="error" id="appointmentDate-error"><?php echo htmlspecialchars($errors['appointmentDate'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="appointmentTime" class="label required">Appointment Time (Fixed at 10:00 AM)</label>
                        <input type="time" id="appointmentTime" name="appointmentTime" required autocomplete="off" min="05:00" max="22:00" value="10:00" readonly aria-required="true">
                        <div class="photo-upload-note">Fixed at 10:00 AM</div>
                        <span class="error" id="appointmentTime-error"><?php echo htmlspecialchars($errors['appointmentTime'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="purpose" class="label required">Purpose of Appointment</label>
                    <input type="text" id="purpose" name="purpose" required readonly autocomplete="off" aria-required="true" value="Orientation" placeholder="Enter the purpose of your appointment">
                    <span class="error" id="purpose-error"><?php echo htmlspecialchars($errors['purpose'] ?? ''); ?></span>
                </div>
            </div>

            <!-- Feedback and Submit -->
            <div class="form-group">
                <div class="success-message" id="successMessage" role="alert"><?php echo htmlspecialchars($success_message); ?></div>
                <div class="error-message" id="errorMessage" role="alert"><?php echo htmlspecialchars($errors['general'] ?? ''); ?></div>
                <div class="loading-spinner" id="loadingSpinner" aria-label="Loading"></div>
            </div>
            <?php if (!empty($debug_log)): ?>
                <div class="debug-log">
                    <h3>Debug Log:</h3>
                    <?php foreach ($debug_log as $log): ?>
                        <p><?php echo htmlspecialchars($log); ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            <div class="form-group" style="text-align: center;">
                <button type="submit" name="submit_appointment" value="1" class="btn btn-primary" aria-label="Submit appointment form">Submit Appointment</button>
            </div>
        </form>
        </div>
<script>
                //merge Address
            const mergeAddress = () => {
            const streetEl   = document.getElementById('street');
            const barangayEl = document.getElementById('barangay');
            const cityEl     = document.getElementById('city');
            const provinceEl = document.getElementById('province');
            const addressEl  = document.getElementById('address');
            const errorEl    = document.getElementById('address-error');

            if (!streetEl || !barangayEl || !cityEl || !provinceEl || !addressEl) return;

            const street   = streetEl.value.trim();
            const barangay = barangayEl.value.trim();
            const city     = cityEl.value.trim();
            const province = provinceEl.value.trim();

            addressEl.value = [street, barangay, city, province].filter(Boolean).join(', ');

            errorEl.textContent =
                (!street || !barangay || !city || !province)
                    ? 'Please fill in all address fields.'
                    : '';
        };

        ['street', 'barangay', 'city', 'province'].forEach(id => {
            const el = document.getElementById(id);
            if (el) {
                el.addEventListener('input', mergeAddress);
            }
        });
</script>
    <script src="https://cdn.jsdelivr.net/npm/flatpickr" defer></script>
    <script src="/assets/js/fillupform1.js" defer></script>
</body>
</html>