document.addEventListener("DOMContentLoaded", () => {
    const popup = document.getElementById("popup-box");
    const message = document.getElementById("popup-message");

    function showPopup(msg, success = true) {
        message.textContent = msg;
        popup.style.display = "block";
        popup.style.background = success ? "#d4edda" : "#f8d7da";
        popup.style.color = success ? "#155724" : "#721c24";
        popup.style.borderColor = success ? "#c3e6cb" : "#f5c6cb";
    }

    function attachDeleteEvent(button, row) {
        button.addEventListener("click", async function () {
            if (!confirm("Are you sure you want to delete this appointment?")) return;

            const appointmentId = button.getAttribute("data-id");

            const deleteForm = new URLSearchParams();
            deleteForm.append("appointment_id", appointmentId);

            const deleteResponse = await fetch("delete_appointment.php", {
                method: "POST",
                headers: { "Content-Type": "application/x-www-form-urlencoded" },
                body: deleteForm.toString()
            });

            const deleteResult = await deleteResponse.json();

            if (deleteResult.success) {
                row.remove();
                showPopup(deleteResult.message);
            } else {
                alert(deleteResult.message || "Failed to delete appointment.");
            }
        });
    }

    // Attach listeners to Accept / Reject buttons
    document.querySelectorAll(".action-btn").forEach(button => {
        const row = button.closest("tr");

        if (button.classList.contains("accept") || button.classList.contains("reject")) {
            button.addEventListener("click", async function () {
                const isAccept = this.classList.contains("accept");
                const status = isAccept ? "Approved" : "Rejected";
                const appointmentId = this.getAttribute("data-id");
                const email = this.getAttribute("data-email");

                if (!confirm(`Are you sure you want to ${status.toLowerCase()} this appointment?`)) return;

                const formData = new URLSearchParams();
                formData.append("appointment_id", appointmentId);
                formData.append("email", email);
                formData.append("status", status);

                const response = await fetch("update_status.php", {
                    method: "POST",
                    headers: { "Content-Type": "application/x-www-form-urlencoded" },
                    body: formData.toString()
                });

                const result = await response.json();

                if (result.success) {
                    showPopup(result.message, true);
                    
                    // Update status visually immediately
                    const statusCell = row.querySelector(".status-cell");
                    statusCell.textContent = status;
                    
                    // Refresh page after 3 seconds
                    setTimeout(() => {
                        window.location.reload();
                    }, 3000);
                } else {
                    showPopup("Error: " + result.message, false);
                }
            });
        } else if (button.classList.contains("delete")) {
            // This is the fix — attach event listeners to existing delete buttons
            attachDeleteEvent(button, row);
        }
    });

    document.getElementById("popup-close").onclick = () => {
        popup.style.display = "none";
    }; 

    
}); 


//search
function performSearch() {
    const searchInput = document.getElementById("searchInput");
    const clearButton = document.getElementById("clearSearch");
    const filter = searchInput.value.toLowerCase().trim();
    
    // Show/hide clear button
    clearButton.style.display = filter ? "block" : "none";

    // Create URL with search parameter
    const url = new URL(window.location.href);
    if (filter) {
        url.searchParams.set('search', filter);
        url.searchParams.set('page', '1'); // Reset to first page when searching
    } else {
        url.searchParams.delete('search');
    }

    // Update URL and reload page
    window.location.href = url.toString();
}

// Search input event listeners
document.getElementById("searchInput").addEventListener("keypress", function(e) {
    if (e.key === "Enter") {
        e.preventDefault();
        performSearch();
    }
});

document.getElementById("searchButton").addEventListener("click", function() {
    performSearch();
});

document.getElementById("clearSearch").addEventListener("click", function() {
    document.getElementById("searchInput").value = "";
    const url = new URL(window.location.href);
    url.searchParams.delete('search');
    window.location.href = url.toString();
});

// Set search input value from URL parameter if it exists
window.addEventListener('load', function() {
    const urlParams = new URLSearchParams(window.location.search);
    const searchValue = urlParams.get('search');
    if (searchValue) {
        const searchInput = document.getElementById("searchInput");
        const clearButton = document.getElementById("clearSearch");
        searchInput.value = searchValue;
        clearButton.style.display = "block";
    }
});

    const paginationLinks = document.querySelectorAll('a[href^="?page="]');
    paginationLinks.forEach(link => {
        link.addEventListener('click', () => window.scrollTo({ top: 0, behavior: 'smooth' }));
    });

    
// View Button Modal Logic
document.querySelectorAll(".view").forEach(button => {
    button.addEventListener("click", function () {
        const data = JSON.parse(this.dataset.info);
        const modal = document.getElementById("view-modal");
        const modalBody = document.getElementById("modal-body");

modalBody.innerHTML = `
    <div style="display:flex; flex-direction:column; align-items:center; gap:15px;">
        <!-- Profile Photo -->
        <img src="${data.profile_photo}" 
             onerror="this.onerror=null;this.src='../Profile_Photo/no-profile.jpg';"
             alt="Profile Photo" 
             style="width:100px; height:100px; border-radius:50%; object-fit:cover; border:2px solid #003087;">

        <p><strong>Name:</strong> ${data.first_name} ${data.last_name}</p>
        <p><strong>Email:</strong> ${data.email}</p>
        <p><strong>Date:</strong> ${data.appointment_date}</p>
        <p><strong>Time:</strong> ${data.appointment_time}</p>
        <p><strong>Contact No.:</strong>+63${data.contact}</p>
        <p><strong>Status:</strong> ${data.status}</p>
        
        <p><strong>ID Type:</strong> ${data.id_type || 'N/A'}</p>
        <p><strong>ID Number:</strong> ${data.id_number || 'N/A'}</p>

        <!-- ID Image -->
        <img src="${data.id_photo || '../IdPhoto/no-id.jpg'}" 
             alt="ID Photo" 
             style="width:100%; max-width:250px; border-radius:6px; box-shadow:0 2px 6px rgba(0,0,0,0.2); border:1px solid #ccc;">
    </div>
`;


        modal.style.display = "flex";
    });
});

document.getElementById("modal-close").onclick = function () {
    document.getElementById("view-modal").style.display = "none";
};

window.onclick = function (e) {
    const modal = document.getElementById("view-modal");
    if (e.target === modal) {
        modal.style.display = "none";
    }
};

// Inline editable remarks logic
document.querySelectorAll(".remarks-text[contenteditable]").forEach(span => {
    span.addEventListener("keydown", async function (e) {
        if (e.key === "Enter") {
            e.preventDefault();
            const appointmentId = this.getAttribute("data-id");
            const remarks = this.textContent.trim();

            // Prevent multiple submissions
            this.blur();

            const formData = new URLSearchParams();
            formData.append("appointment_id", appointmentId);
            formData.append("remarks", remarks);

            try {
                const response = await fetch("update_remarks.php", {
                    method: "POST",
                    headers: { "Content-Type": "application/x-www-form-urlencoded" },
                    body: formData.toString()
                });
                const result = await response.json();

                if (result.success) {
                    // Show popup and reload after short delay
                    document.getElementById("popup-message").textContent = "Remarks updated!";
                    document.getElementById("popup-box").style.display = "block";
                    setTimeout(() => {
                        window.location.reload();
                    }, 800);
                } else {
                    document.getElementById("popup-message").textContent = "Failed to update remarks.";
                    document.getElementById("popup-box").style.display = "block";
                }
            } catch (err) {
                document.getElementById("popup-message").textContent = "Network error.";
                document.getElementById("popup-box").style.display = "block";
            }
        }
    });
});

document.querySelectorAll('.remarks-text').forEach(span => {
    if (span.textContent.trim() === "") {
        span.textContent = "";
    }
});

// Report Modal Logic
const reportModal = document.getElementById("reportModal");
const downloadReportBtn = document.getElementById("downloadReportBtn");
const reportModalClose = document.getElementById("reportModalClose");

downloadReportBtn.onclick = function() {
    reportModal.style.display = "flex";
}

reportModalClose.onclick = function() {
    reportModal.style.display = "none";
}

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target === reportModal) {
        reportModal.style.display = "none";
    }
    if (event.target === document.getElementById("view-modal")) {
        document.getElementById("view-modal").style.display = "none";
    }
}

// Set default month to current month
document.getElementById("reportMonth").value = new Date().toISOString().slice(0,7);

// Add this function after your existing code
function generateClientForm(data) {
    // Redirect to generate_form.php with the appointment ID
    window.location.href = `generate_form.php?id=${data.appointment_id}`;
}


document.querySelectorAll(".generate").forEach(button => {
    button.addEventListener("click", function() {
        const data = JSON.parse(this.dataset.info);
        generateClientForm(data);
    });
});