document.addEventListener('DOMContentLoaded', () => {
    const form = document.getElementById("verifyForm");
    const otpInput = document.getElementById("otp");
    const otpError = document.getElementById("otp-error");
    const submitButton = document.getElementById("submit-btn");
    const resendLink = document.getElementById("resend-otp");
    const timerDisplay = document.getElementById("otp-timer");
    const expiredMessage = document.getElementById("expired-message");
    const successMessage = document.getElementById("success-message");

    // Get OTP time from data attribute (set this in your PHP file)
    const otpTime = timerDisplay?.dataset.otpTime;
    
    // OTP expiration timer
    if (otpTime && !successMessage?.style.display === "block" && !expiredMessage?.style.display === "block") {
        let timeLeft = 180 - (Math.floor(Date.now() / 1000) - parseInt(otpTime));
        if (timeLeft > 0) {
            const timerInterval = setInterval(() => {
                if (timeLeft <= 0) {
                    clearInterval(timerInterval);
                    if (timerDisplay) timerDisplay.textContent = "OTP has expired.";
                    if (expiredMessage) {
                        expiredMessage.style.display = "block";
                        expiredMessage.textContent = 'OTP has expired. Please click "Resend OTP" to receive a new code.';
                    }
                    if (otpInput) otpInput.value = '';
                    if (otpError) otpError.textContent = '';
                    if (submitButton) submitButton.disabled = false;
                } else {
                    if (timerDisplay) timerDisplay.textContent = `OTP expires in ${timeLeft} seconds`;
                    timeLeft--;
                }
            }, 1000);
        } else {
            if (timerDisplay) timerDisplay.textContent = "OTP has expired.";
            if (expiredMessage) {
                expiredMessage.style.display = "block";
                expiredMessage.textContent = 'OTP has expired. Please click "Resend OTP" to receive a new code.';
            }
            if (otpInput) otpInput.value = '';
            if (otpError) otpError.textContent = '';
            if (submitButton) submitButton.disabled = false;
        }
    }

    // Input validation
    otpInput?.addEventListener("input", () => {
        otpInput.value = otpInput.value.replace(/[^0-9]/g, '').slice(0, 6);
        if (otpError) otpError.textContent = '';
        if (expiredMessage?.style.display === "block") {
            expiredMessage.style.display = "none";
        }
    });

    // Form submission
    form?.addEventListener("submit", (e) => {
        e.preventDefault();
        if (otpError) otpError.textContent = '';

        if (!/^\d{6}$/.test(otpInput?.value || '')) {
            if (otpError) otpError.textContent = 'Please enter a valid 6-digit OTP.';
            return;
        }

        if (submitButton) {
            submitButton.disabled = true;
            submitButton.classList.add('loading');
        }
        form.submit();
    });

    // Resend OTP timer with sessionStorage persistence
    const RESEND_COOLDOWN = 60; // 60 seconds cooldown
    let resendCooldown = parseInt(sessionStorage.getItem('resendCooldown') || '0');
    let resendTimerStart = parseInt(sessionStorage.getItem('resendTimerStart') || '0');

    function startResendTimer() {
        if (!resendLink) return;

        resendLink.classList.add('disabled');
        const now = Math.floor(Date.now() / 1000);
        resendTimerStart = now;
        resendCooldown = RESEND_COOLDOWN;
        sessionStorage.setItem('resendTimerStart', resendTimerStart.toString());
        sessionStorage.setItem('resendCooldown', resendCooldown.toString());

        const resendInterval = setInterval(() => {
            const elapsed = Math.floor(Date.now() / 1000) - resendTimerStart;
            resendCooldown = RESEND_COOLDOWN - elapsed;

            if (resendCooldown <= 0) {
                clearInterval(resendInterval);
                if (resendLink) {
                    resendLink.classList.remove('disabled');
                    resendLink.textContent = 'Resend OTP';
                }
                sessionStorage.removeItem('resendCooldown');
                sessionStorage.removeItem('resendTimerStart');
            } else {
                if (resendLink) resendLink.textContent = `Resend OTP (${resendCooldown}s)`;
                sessionStorage.setItem('resendCooldown', resendCooldown.toString());
            }
        }, 1000);
    }

    // Check if the timer should resume on page load
    if (resendTimerStart > 0) {
        const elapsed = Math.floor(Date.now() / 1000) - resendTimerStart;
        resendCooldown = RESEND_COOLDOWN - elapsed;
        if (resendCooldown > 0) {
            startResendTimer();
        } else {
            sessionStorage.removeItem('resendCooldown');
            sessionStorage.removeItem('resendTimerStart');
        }
    }

    // Resend link click handler
    resendLink?.addEventListener("click", (e) => {
        if (resendLink.classList.contains('disabled')) {
            e.preventDefault();
        } else {
            startResendTimer();
        }
    });

    // Handle success message redirect
    if (successMessage?.style.display === "block") {
        setTimeout(() => {
            window.location.href = '../index.php';
        }, 2000);
    }
});
